import { Box, Button, FormControl, FormControlLabel, FormLabel, Radio, RadioGroup, Typography } from "@mui/material";
import jwtDecode from "jwt-decode";
import { useEffect, useMemo, useState } from "react";
import nacl from "tweetnacl";
import { getTenentVoting, postVote } from "../../actions/data-manager";
import { JWT_TENANT } from "../../actions/network-manager";
import { VotingEntity } from "../../internal-types";
import { keyToBase64, PRIVATE_KEY } from "../register/RegisterForm";
import { Warning } from "./Warning";
import { Buffer } from "buffer";
import { TabWrapper } from "../TabWrapper/TabWrapper";
import { Card } from "../Card/Card";

const classes = {
    wrapper: {
        display: 'flex',
        flexDirection: 'column',
        gap: '50px',
    },
    header: {
        fontWeight: 600,
        fontSize: '1.6em',
    },
    subheader: {
        color: 'gray',
        fontSize: '0.9rem'
    },
    subject: {
        fontSize: '1.2rem'
    },
    button: {
        marginLeft: 'auto',
        borderRadius: '50px'
    }
} as const;

export const Voting = ({ buildingId }: { buildingId: string }): JSX.Element => {

    const [status, setStatus] = useState();
    const [voting, setVoting] = useState<VotingEntity>();
    const [selectedCandidateId, setSelectedCandidateId] = useState('');
    const [errorText, setErrorText] = useState('');
    const [privateKey, setPrivateKey] = useState(localStorage.getItem(PRIVATE_KEY));

    useEffect(() => {
        getTenentVoting(buildingId, (err: any, res: any, disp: any) => {
            setStatus(res.status || "not allowed");
            setVoting(res);
        })
    }, []);

    const handleVote = () => {
        let voteData = { votingId: voting!._id, votedFor: selectedCandidateId, time: new Date() };
        let voteDataStr = JSON.stringify(voteData);
        let dataBuffer = Buffer.from(voteDataStr);

        let buffKey = Buffer.from(privateKey!, 'base64');
        const signatureRaw = nacl.sign.detached(dataBuffer, buffKey);
        let signature = keyToBase64(signatureRaw);

        let vote = { vote: voteDataStr, signature };

        postVote(buildingId, vote, (err: any, data: any) => {
            if (err) {
                if (err.status === 406) {
                    setErrorText("Signature not valid!");
                } else {
                    setErrorText(err.errorMessage);
                }
            } else {
                setStatus(data.status)
            }

        })(null);



    }

    return (<>
        {
            !!voting &&
            <TabWrapper header="Glasanje">
                {!!privateKey &&
                    <>
                        {status === "not allowed" && <Warning text='Vaš nalog nije potvrđen kao vlasnik ni jedne jedinice. Molimo obratite se upravniku stambene zajednice.' />}
                        {status !== "not allowed" &&
                            <Card>
                                <Typography sx={classes.subheader}>Tekuće glasanje</Typography>
                                {status === "no voting" && <Typography>U ovom trenutku nema aktivnog glasanja.</Typography>}
                                {status === "in progress" &&
                                    <>
                                        <Typography sx={classes.subject}>{voting.subject}</Typography>
                                            <FormControl>
                                                <FormLabel>Opcije</FormLabel>
                                                <RadioGroup value={selectedCandidateId} onChange={(event) => { setSelectedCandidateId(event.target.value) }}>
                                                    {voting.candidates.map((candidate, ind: number) => (<FormControlLabel value={candidate._id} control={<Radio size="small"/>} label={candidate.name} key={ind} />))}
                                                </RadioGroup>
                                            </FormControl>
                                        <Button variant='contained' sx={classes.button} disabled={selectedCandidateId === ''} onClick={handleVote}>Glasaj</Button>
                                    </>
                                }
                                {status === "voted" && <Typography>Hvala vam na glasanju. </Typography>}
                            </Card>
                        }
                    </>
                }
                {
                    !privateKey && <Warning text='Glasanje je onemogućeno jer privatan ključ nije sačuvan u aplikaciji. Molimo vas, podesite ključ u podešavanjima!' />
                }
            </TabWrapper>
        }
    </>
    );
}